(* ------------------- PBMIN.INC -----------------------

  Include file to load a screen dump from a PBM (portable bitmap) file.
  Supported resolutions: 720x256, 720x512
  Provides the following functions:

    procedure LoadScreenFromPBM -> Get image row from a PBM file
                                   and copy it to the Roller RAM

  Uses include file KERNEL.INC.

  ---------------------------------------------------- *)
const
  BufSize = 128;

type
  TFileName  = string(.14.);
  BinaryFile = file;
  ErrorCode  = (ErrOk, ErrFileNotFound, ErrFormatNotSupported);

var
  fbuf:         Array(.1..BufSize.) of byte;
  fbufidx:      byte;
  recsread:     integer;
  pbmfile:      BinaryFile;
  headerOk:     boolean;
  doubleHeight: boolean;
  result:       ErrorCode;


(* Reads a byte from the buffer. Reads a block from disc
   as soon as the buffer is read to the end. *)
function ReadBuf: byte;
begin
  if fbufidx = BufSize then begin
    fbufidx := 1;
    BlockRead(pbmfile, fbuf, 1, recsread); (* reads a block of 128 bytes *)
  end
  else
    fbufidx := succ(fbufidx);
  ReadBuf := fbuf(.fbufidx.);
end; (* ReadBuf *)

(* Checks whether the next value from the buffer equals
   the byte in the parameter. *)
function EqualsNextBufVal(b: byte): boolean;
begin
    EqualsNextBufVal := (ReadBuf = b);
end; (* EqualsNextBufVal *)


(* Verifies the 'magic number', which should be 'P4' at the beginning of
   the file. *)
function IsMagicNumberValid: boolean;
var result: boolean;
begin
  result := EqualsNextBufVal($50) and (* 'P' *)
            EqualsNextBufVal($34) and (* '4' *)
            EqualsNextBufVal($0A);    (* LF  *)
  if recsread = 0 then result := false; (* out of data *)
  
  IsMagicNumberValid := result;
end; (* IsMagicNumberValid *)


(* Skips a comment line. If none was found, false is returned. *)
function SkipCommentLine: boolean;
var
  result:   boolean;
  b:        byte;
  readMore: boolean;
begin
  result := false;
  b := ReadBuf;
  if recsread > 0 then begin
    if b = $23 then begin (* '#' *)
      result := true;     (* comment line found *)
      readMore := true;
      while readMore do begin
        readMore := not EqualsNextBufVal($0A);  (* LF = end of comment line *)
        if recsread = 0 then begin (* out of data *)
          readMore := false;
          result := false;
        end;
      end;
    end else begin
      (* set back buffer index, so the byte gets read again *)
      fbufidx := pred(fbufidx);
    end;
  end;
  
  SkipCommentLine := result;
end; (* SkipCommentLine *)


(* Reads and returns a resolution value (X or Y). *)
function ReadRes: integer;
var
  b:          byte;
  readMore:   boolean;
  res:        integer;
  digitCount: integer;
begin
  res := 0;
  digitCount := 0;
  readMore := true;
  
  while readMore do begin
    b := ReadBuf;
    if recsread = 0 then begin
      readMore := false; (* out of data *)
    end else begin
      if (b = $20) or (b = $0A) then begin  (* ' ' or LF *)
        readMore := false;
      end else begin
        res := 10 * res + b - ord('0');     (* build the number from digits *)
        digitCount := succ(digitCount);
        if digitCount > 3 then begin        (* resolutions with more than     *)
          res := 0;                         (* three digits are not supported *)
          readMore := false;
        end;
      end;
    end;
  end;
  
  ReadRes := res;
end; (* ReadRes *)


(* Reads a line of pixels. If invert is true, then all pixels get inverted. *)
procedure ReadPixelLine(y: byte; invert: boolean);
var
  x: byte;
  i: integer;
  b: byte;
begin
  i := y;
  for x := 1 to 90 do begin (* 90 = 720 / 8 *)
    b := ReadBuf;
    if invert then b := b xor $FF;
    GX_Buffer(.i.) := b;
    i := i + 8;
  end;
end; (* ReadPixelLine *)


(* Loads a screen dump from a black and white 1 bit per pixel PBM file. *)
function LoadScreenFromPBM(filename: TFileName; invert: boolean): ErrorCode;
var
  row:  byte;
  x, y: byte;
  i:    integer;
  b:    byte;
  resX: integer;
  resY: integer;
begin
  fbufidx := BufSize;
  headerOk := false;
  doubleHeight := false;

  Assign(pbmfile, filename);
  (*$I-*)
  Reset(pbmfile);
  (*$I+*)
  if IOResult <> 0 then begin
    result := ErrFileNotFound;
  end else begin
    (* Read header *)
    if IsMagicNumberValid then begin
      while SkipCommentLine do begin
      end;
      resX := ReadRes;
      if resX > 0 then resY := ReadRes;
      if resY > 0 then begin
        if (resX = 720) and ((resY = 256) or (resY = 512)) then begin
          headerOk := true;
          if resY = 512 then doubleHeight := true;
        end;
      end;
    end;

    if headerOk then begin
      (* Read pixel data *)
      for row := 0 to 31 do begin
        for y := 0 to 7 do begin
          ReadPixelLine(y, invert);
          if doubleHeight then ReadPixelLine(y, invert);
        end;
        PutScreenRow(row); (* puts 720 bytes = 8 rows of 90 bytes *)
      end;
      result := ErrOk;
    end else begin
      result := ErrFormatNotSupported;
    end;
    
    Close(pbmfile);
  end;
  
  LoadScreenFromPBM := Result;
end; (* LoadScreenFromPBM *)
