(* ------------------- PBMOUT.INC ----------------------

  Include file to save a screen dump to a PBM (portable bitmap) file.
  Provides the following functions:

    procedure SaveScreenToPBM -> Get image row from the Roller RAM
                                 and save it to a PBM file

  Uses include file KERNEL.INC.

  ---------------------------------------------------- *)
const
  BufSize = 128;

type
  TFileName  = string(.14.);
  BinaryFile = file;

var
  fbuf   : Array(.1..BufSize.) of byte;
  fbufidx: byte;
  pbmfile: BinaryFile;


(* Writes a byte to the buffer. Writes a block to disc
   as soon as the buffer is full. *)
procedure WriteBuf(b: byte);
begin
  (* WriteLn('WriteBuf: ', b, '  fbufidx=', fbufidx); *)
  fbuf(.fbufidx.) := b;
  if fbufidx = BufSize then begin
    fbufidx := 1;
    BlockWrite(pbmfile, fbuf, 1); (* writes a block of 128 bytes *)
  end
  else
    fbufidx := succ(fbufidx);
end; (* WriteBuf *)


(* Finishes the buffer, i.e. writes the remaining data in
   the buffer to disc. *)
procedure FinishBuf;
  var b: byte;
begin
  if fbufidx > 1 then begin
    for b := fbufidx to BufSize do (* fill rest of fbuf with zeroes *)
      fbuf(.b.) := 0;
    BlockWrite(pbmfile, fbuf, 1); (* writes a block of 128 bytes *)
    fbufidx := 1;
  end;
end; (* FinishBuf *)


(* Writes a line of pixels *)
procedure WritePixelLine(y: byte);
var
  x: byte;
  i: integer;
begin
  i := y;
  for x := 1 to 90 do begin (* 90 = 720 / 8 *)
    WriteBuf(GX_Buffer(.i.));
    i := i + 8;
  end;
end; (* WritePixelLine *)


(* Saves a screen dump to a black and white 1 bit per pixel PBM file. *)
procedure SaveScreenToPBM(filename: TFileName; doubleHeight: boolean);
var
  row      : byte;
  x, y     : byte;
  i        : integer;
begin
  fbufidx := 1;

  Assign(pbmfile, filename);
  (*$I-*)
  Rewrite(pbmfile);
  (*$I+*)
  if IOResult = 0 then begin
    (* Write header *)
    WriteBuf($50); (* 'P' *)
    WriteBuf($34); (* '4' *)
    WriteBuf($0A); (* LF  *)
    WriteBuf($37); (* '7' *)
    WriteBuf($32); (* '2' *)
    WriteBuf($30); (* '0' *)
    WriteBuf($20); (* ' ' *)
    if doubleHeight then begin
      WriteBuf($35); (* '5' *)
      WriteBuf($31); (* '1' *)
      WriteBuf($32); (* '2' *)
    end else begin
      WriteBuf($32); (* '2' *)
      WriteBuf($35); (* '5' *)
      WriteBuf($36); (* '6' *)
    end;
    WriteBuf($0A); (* LF  *)
  
    (* Write pixel data *)
    for row := 0 to 31 do begin
      GetScreenRow(row); (* gets 720 bytes = 8 rows of 90 bytes *)
      for y := 0 to 7 do begin
        WritePixelLine(y);
        if doubleHeight then begin
          WritePixelLine(y);
        end;
      end;
    end;
    
    FinishBuf;
    Close(pbmfile);
  end;
end; (* SaveScreenToPBM *)
