(* ------------------------- INPUT.INC ----------------------------------------- *)
(* Version 1.1                                                                   *)
(*  Include-File fuer Eingabebefehle auf der JOYCE unter Turbo Pascal.           *)
(*  Es sind enthalten:                                                           *)
(*     procedure WriteKeyMessage    -> Schreibt eine Tastendruck-Meldung         *)
(*     procedure WaitForKey         -> Wartet auf einen Tastendruck              *)
(*     procedure Beep               -> Erzeugt einen Ton                         *)
(*     function  InputChar          -> Eingabe eines einzelnen Zeichens          *)
(*     function  InputCharNoEcho    -> Eingabe eines einzelnen Zeichen ohne Echo *)
(*     function  InputLoop          -> Eingabeschleife (intern)                  *)
(*     procedure InputLine          -> Eingabe einer Zeile                       *)
(*     procedure InputFileName      -> Eingabe eines Dateinamens (8.3)           *)
(*     function  InputNaturalNumber -> Eingabe einer natuerlichen Zahl           *)
(*     function  InputInteger       -> Eingabe einer ganzen Zahl                 *)
(*     function  InputReal          -> Eingabe einer reellen Zahl                *)
(*     function  InputExpReal       -> Eingabe einer reellen Zahl mit Exponent   *)
(*     function  InputDate          -> Eingabe eines Datums                      *)
(* ----------------------------------------------------------------------------- *)

(*$V-*)

type
  WorkString = string(.255.);
  FileNameString = string(.14.);
  DateString = string(.10.);

const
  MODE_TEXT        = 1;
  MODE_NUM_NATURAL = 2;
  MODE_INTEGER     = 3;
  MODE_REAL        = 4;
  MODE_EXP_REAL    = 5;
  MODE_DATE        = 6;


(* Writes a 'Press a key' message to the screen. *)
procedure WriteKeyMessage;
begin
  Write('<Press a key>');
end;


(* Waits for a key to be pressed. *)
procedure WaitForKey;
var ch: Char;
begin
  repeat until KeyPressed;
  Read(kbd, ch);
end;


(* Emits a beep through the loudspeaker. *)
procedure Beep;
begin
  Write(chr(7));
end; (* Beep *)


(* Inputs a single character, writes the character to the screen. *)
function InputChar: Char;
var ch: Char;
begin
  Read(kbd, ch);
  if ch = chr(127) then
    Write(chr(8), ' ', chr(8)) (* backspace *)
  else
    Write(ch);    (* echo to screen *)
  InputChar := ch;
end; (* InputChar *)


(* Inputs a single character, no echo to the screen. *)
function InputCharNoEcho: Char;
var ch: Char;
begin
  Read(kbd, ch);
  InputCharNoEcho := ch;
end; (* InputCharNoEcho *)


(* Input loop to input a string of characters depending on the mode.
   Input is finished with one of the following keys:
     Enter, Tab, Down, Up *)
function InputLoop(maxlen: Integer; mode: byte): WorkString;
var ch:  Char;
    cnt: Integer;
    s:   WorkString;
begin
  cnt := 0;
  s := '';
  repeat
    ch := InputChar;
    if ch = chr(127) then begin (* backspace *)
      Delete(s, cnt, 1);
      cnt := pred(cnt);
    end else
      if (cnt < maxlen) then begin
        case mode of
          MODE_TEXT: begin
            if (ch > chr(31)) and (ch < chr(126)) then begin
              s := s + ch;
              cnt := succ(cnt);
            end else begin
              Write(chr(8));
            end;
          end;
          MODE_NUM_NATURAL: begin
            if (ch >= '0') and (ch <= '9') then begin
              s := s + ch;
              cnt := succ(cnt);
            end else begin
              Write(chr(8));
            end;
          end;
          MODE_INTEGER: begin
            (* TODO *)
          end;
          MODE_REAL: begin
            (* TODO *)
          end;
          MODE_EXP_REAL: begin
            (* TODO *)
          end;
          MODE_DATE: begin
            (* TODO *)
          end;
        end; (* case *)
      end else begin
        Write(chr(8), ' ', chr(8));
        Beep;
      end
  until (ch = chr(13))  (* Enter *)
     or (ch = chr(9))   (* Tab   *)
     or (ch = chr(30))  (* Down  *)
     or (ch = chr(31)); (* Up    *)
  
  InputLoop := s;
end; (* InputLoop *)


(* Inputs a line of text; input is finished with one of the following keys:
     Enter, Tab, Down, Up
   Backspace can be used to correct the input.   *)
function InputLine(maxlen: Integer): WorkString;
begin
  InputLine := InputLoop(maxlen, MODE_TEXT);
end;


(* Inputs an 8.3 filename with optional drive letter; input is finished with
   one of the following keys: Enter, Tab, Down, Up.
   Backspace can be used to correct the input. *)
function InputFileName: FileNameString;
var ch:           Char;
    cntFilename:  Integer;
    cntExtension: Integer;
    dotEntered:   Boolean;
    colonEntered: Boolean;
    len:          Integer;
    s:            WorkString;
    ch1:          Char;
begin
  cntFilename := 0;
  cntExtension := 0;
  dotEntered := false;
  colonEntered := false;
  s := '';
  repeat
    ch := InputChar;
    len := Length(s);
    case Ord(ch) of
      127: (* backspace *)
        begin
          if len > 0 then begin
            case s(.len.) of
              '.': dotEntered := false;
              ':': colonEntered := false
              else begin
                if dotEntered then cntExtension := pred(cntExtension)
                else cntFilename := pred(cntFilename);
              end;
            end; (* case *)
            Delete(s, len, 1);
          end
          else Write(' '); (* undo cursor left *)
        end; (* 127 *)
      58: (* ':' *)
        begin
          if not colonEntered and (len = 1) then begin
            colonEntered := true;
            s := s + ch;
          end
          else Write(chr(8), ' ', chr(8));
        end; (* 58 *)
      46: (* '.' *)
        begin
          if not dotEntered and ((colonEntered and (len > 2)
                           or not colonEntered and (len > 0))) then begin
            dotEntered := true;
            s := s + ch;
          end
          else Write(chr(8), ' ', chr(8));
        end (* 46 *)
      else begin
        if (ch >= 'A') and (ch <= 'Z') or
           (ch >= 'a') and (ch <= 'z') or
           (ch >= '0') and (ch <= '9') or
           (ch = '_') then begin (* allowed characters *)
          if dotEntered then begin
            if cntExtension < 3 then begin
              s := s + ch;
              cntExtension := succ(cntExtension);
            end
            else Write(chr(8), ' ', chr(8));
          end
          else begin (* dotEntered = false *)
            if cntFilename < 9 then begin
              s := s + ch;
              cntFilename := succ(cntFilename);
            end
            else Write(chr(8), ' ', chr(8));
          end;
        end
        else Write(chr(8), ' ', chr(8)); (* not allowed *)
      end; (* else *)
    end; (* case *)
  until (ch = chr(13))  (* Enter *)
     or (ch = chr(9))   (* Tab   *)
     or (ch = chr(30))  (* Down  *)
     or (ch = chr(31)); (* Up    *)
  
  InputFileName := s;
end; (* InputFileName *)


(* Inputs a natural number until the Enter key is pressed. *)
(* Backspace can be used to correct the input.             *)
function InputNaturalNumber(maxlen: byte): Integer;
var s:    WorkString;
    num:  Integer;
    code: Integer;
begin
  s := InputLoop(maxlen, MODE_NUM_NATURAL);
  WriteLn; WriteLn('s = ', s); (* TEST *)
  Val(s, num, code);
  if code <> 0 then
    num := 0;
  WriteLn; WriteLn('code = ', code); (* TEST *)

  InputNaturalNumber := num;
end; (* InputNaturalNumber *)

function InputInteger(maxlen: byte): Integer;
begin
  InputInteger := 0;
end; (* InputInteger *)

function InputReal(maxlen: byte): Integer;
begin
  InputReal := 0;
end; (* InputReal *)

function InputExpReal(maxlen: byte): Integer;
begin
  InputExpReal := 0;
end; (* InputExpReal *)

function InputDate(maxlen: byte): DateString;
begin
  InputDate := '01.01.1900';
end; (* InputDate *)

(* ------------------ Ende von INPUT.INC ----------------------- *)
