(* HP-GL Interpreter Version 3.5 *)
(* Reads HP-GL (usually *.plt) files and draws the graphics on screen. *)
program HpglInterpreter;
(*$IKERNEL2.INC*)
(*$IINPUT.INC*)
(*$IBOX.INC*)
(*$ISCREEN2.INC*)
(*$IPBMOUT.INC*)
label
  byebye;
const
  MODE_SET  = 0;
  PAUSE_MODE = true;
  START     = 'S';
  MNEMONIC  = 'M';
  PARAMETER = 'P';
  ERROR     = 'E';
  UNKNOWN   = 0;
  INIT      = 1;
  PEN_DOWN  = 2;
  PEN_UP    = 3;
  POS_ABS   = 4;
  POS_REL   = 5;
  MIN_COORD = -10000;
  MAX_COORD =  10000;
var
  Break:       Boolean;
  FilVar:      Text;
  Ch:          Char;
  State:       Char;
  I:           Integer;
  FileName:    FileNameString;
  FileNameCfg: FileNameString;
  FileNamePBM: FileNameString;
  Command:     Integer;
  FirstMnemCh: Char;
  Value:       Integer;
  IsNegative:  Boolean;
  IsPenDown:   Boolean;
  LastX:       Integer;
  LastY:       Integer;
  ParamX:      Integer;
  ParamY:      Integer;
  ParamName:   Char;
  DotCnt:      Integer;
  MaxX, MaxY:  Integer;
  MinX, MinY:  Integer;
  OffsetX:     Integer;
  OffsetY:     Integer;
  ScaleX:      Real;
  ScaleY:      Real;
  Pass:        Integer;
  Row:         Byte;
  CreatePbm:   Boolean;
  ARCorr:      Char;

(* forward declarations *)

(* procedure PrintCommand;   forward;  Test *)
procedure ScaleParameter; forward;
procedure Draw;           forward;


(* functions and procedures *)

(* Initializes a parameter, i.e. resets all necessary variables
   before a new parameter can be read in. *)
procedure InitParameter;
begin
  Value := 0;
  IsNegative := false;
end; (* InitParameter *)


(* Checks if a character is a whitespace. *)
function isWhitespace(c: Char): Boolean;
begin
  isWhitespace := ord(c) < 33;
end; (* isWhitespace *)


(* Finishes a parameter, i.e. all the work that has to be done after
   an HP-GL parameter has been read in completely. *)
procedure FinishParameter;
begin
  if (Command > INIT) and (Command < 6) then begin
    if IsNegative then Value := -Value;
    if ParamName = 'X' then begin
      LastX := ParamX;
      ParamX := Value;
      if (Pass = 1) then begin
        if Value > MaxX then MaxX := Value;
        if Value < MinX then MinX := Value;
      end;
      ParamName := 'Y';
    end else begin
      LastY := ParamY;
      ParamY := Value;
      ScaleParameter;
      if (Pass = 1) then begin
        if Value > MaxY then MaxY := Value;
        if Value < MinY then MinY := Value;
      end;
      ParamName := 'X';
      if (Pass = 2) and IsPenDown then Draw;
    end;
  end;
  InitParameter;
end; (* FinishParameter *)


(* Scales the X and Y HP-GL parameters, transforming them to screen
   coordinates. *)
procedure ScaleParameter;
var r: Real;
begin
  r := ParamX - MinX;
  r := r * ScaleX;
  ParamX := trunc(r) + OffsetX;

  r := MaxY - ParamY;
  r := r * ScaleY;
  ParamY := trunc(r) + OffsetY;
end; (* ScaleParameter *)


(* Prints a dot every 10 rounds. *)
procedure PrintDot;
begin
  if Pass = 1 then begin
    DotCnt := pred(DotCnt);
    if DotCnt = 0 then begin
      DotCnt := 10;
      Write('.');
    end;
  end;
end; (* PrintDot *)


(* Processes a character depending of the current state of the state machine. *)
procedure ProcessChar;
begin
  if Ch = ';' then begin
    State := START;
    FinishParameter;
    ParamName := 'X';
  end else begin
    case State of
      START: begin
          PrintDot;
          Command := UNKNOWN;
          InitParameter;
          if not isWhitespace(Ch) then begin
            if (Ch >= 'A') and (Ch <= 'Z') then begin
              FirstMnemCh := Ch;
              State := MNEMONIC;
            end else begin
              WriteLn;
              WriteLn('Invalid first character in Mnemonic: ', Ch);
              State := ERROR;
            end;
          end;
        end;
      MNEMONIC: begin
          if (Ch >= 'A') and (Ch <= 'Z') then begin
            case FirstMnemCh of
              'I': case Ch of
                     'N': begin Command := INIT;     IsPenDown := false; end;
                   end;
              'P': case Ch of
                     'D': begin Command := PEN_DOWN; IsPenDown := true;  end;
                     'U': begin Command := PEN_UP;   IsPenDown := false; end;
                     'A': begin Command := POS_ABS;                      end;
                     'R': begin Command := POS_REL;                      end;
                   end;
            end;
            (* PrintCommand;  Test *)
            State := PARAMETER;
          end else begin
            WriteLn;
            WriteLn('Invalid second character in Mnemonic: ', Ch);
            State := ERROR;
          end;
        end;
      PARAMETER: begin
          if Ch <> '.' then begin (* real values not supported; dot ignored *)
            if (Ch = '-') then begin
              IsNegative := true;
            end else if (Ch >= '0') and (Ch <='9') then begin
              Value := Value * 10 + Ord(Ch) - Ord('0');
            end else begin
              (* delimiter found: end parameter *)
              FinishParameter;
            end;
          end;
        end;
    end; (* case *)
  end;
end; (* ProcessChar *)


(* TEST >>>
procedure PrintCommand;
begin
  case Command of
    UNKNOWN:  Write('UNKNOWN');
    PEN_DOWN: Write('PEN_DOWN');
    PEN_UP:   Write('PEN_UP');
    POS_ABS:  Write('POS_ABS');
    POS_REL:  Write('POS_REL');
  end;
end; ( PrintCommand <<< TEST *)


(* Draws a line; includes a parameter check to prevent overflow. *)
procedure Draw;
begin
  if (LastX >= 0) and (LastX < 720) and
     (LastY >= 0) and (LastY < 256) and
     (ParamX >= 0) and (ParamX < 720) and
     (ParamY >= 0) and (ParamY < 256) then begin
    line(LastX, LastY, ParamX, ParamY, MODE_SET);
  end;
end; (* Draw *)


(* Calculates the scaling of the picture with the correct aspect ratio,
   taking into account the correct pixel aspect ratio. *)
procedure CalcScaling;
var height, width: Integer;
    h2, w2: Real;
begin
  width  := MaxX - MinX;
  height := MaxY - MinY;
  if height > width then begin
    ScaleX := 540 / height;
    ScaleY := ScaleX * 256 / 540;
    w2 := width * ScaleX;
    OffsetX := (720 - trunc(w2)) div 2;
    OffsetY := 0;
  end else begin
    ScaleX := 720 / width;
    ScaleY := ScaleX * 256 / 540;
    h2 := height * ScaleY;
    OffsetX := 0;
    OffsetY := (256 - trunc(h2)) div 2;
  end;
end; (* CalcScaling *)


(* Creates a filename for the configuration file. It has the same name as
   the HP-GL file, but width the extension '.cfg'. *)
procedure CreateConfigFileName;
var i: Integer;
    ch: Char;
    len: Integer;
begin
  i := 1;
  len := length(filename);
  FileNameCfg := '';

  repeat (* copy filename before the dot *)
    ch := FileName(.i.);
    FileNameCfg := FileNameCfg + ch;
    i := succ(i);
  until (i > len) or (ch = '.');

  if ch = '.' then begin (* add cfg extension *)
    FileNameCfg := FileNameCfg + 'cfg';
  end;
end; (* CreateConfigFileName *)


(* First pass: acquire mininum and maximum parameter values. *)
procedure Pass1;
var filVarCfg: Text;
begin
  Pass := 1;
  
  WriteLn('First pass: determining min. and max. X and Y coordinates.');
  
  CreateConfigFileName;

  Assign(filVarCfg, FileNameCfg);
  (*$I-*)
  Reset(filVarCfg);
  (*$I+*)
  if IOResult <> 0 then
  begin
    WriteLn('Configuration file not found: ', FileNameCfg);
    WriteKeyMessage;
    WaitForKey;
    (* open cfg file *)
    Rewrite(filVarCfg);
	
	(* get min and max parameter values *)
    while (not Eof(FilVar)) and (State <> ERROR) do begin
      Read(FilVar, Ch);
      ProcessChar;
    end; (* while *)
	
	(* write min and max values to configuration file *)
    WriteLn(filVarCfg, MinX);
    WriteLn(filVarCfg, MaxX);
    WriteLn(filVarCfg, MinY);
    WriteLn(filVarCfg, MaxY);
  end
  else begin
    (* read cfg file *)
    ReadLn(filVarCfg, MinX);
    ReadLn(filVarCfg, MaxX);
    ReadLn(filVarCfg, MinY);
    ReadLn(filVarCfg, MaxY);
    WriteLn;
    WriteLn('Configuration read:');
    WriteLn('Min. X: ', MinX:6, '   Max. X: ', MaxX:6);
    WriteLn('Min. Y: ', MinY:6, '   Max. Y: ', MaxY:6);
    WriteKeyMessage;
    WaitForKey;
  end;

  Close(filVarCfg);
end; (* Pass1 *)


(* Second pass: draw the picture. *)
procedure Pass2;
begin
  Pass := 2;
  
  ClrScr;
  WriteLn('Second pass: drawing');
  WriteLn;
  WriteLn('Min. X: ', MinX:6, '   Max. X: ', MaxX:6);
  WriteLn('Min. Y: ', MinY:6, '   Max. Y: ', MaxY:6);
  WriteLn;
  CalcScaling;
  WriteLn('Scale  X: ', ScaleX:6:3, '   Scale  Y: ', ScaleY:6:3);
  WriteLn('Offset X: ', OffsetX:6,  '   Offset Y: ', OffsetY:6);
  WriteKeyMessage;
  WaitForKey;

  Reset(FilVar);
  ClrScr;

  while (not Eof(FilVar)) and (State <> ERROR) do begin
    Read(FilVar, Ch);
    ProcessChar;
  end; (* while *)
end; (* Pass2 *)


(* Main program *)
begin
  ClrScr;
  GraphInit;
  ClearScreenRow(31); (* clears the status line *)
  
  DrawBox(1, 1, 90, 3); GotoXY(27, 2);
  Write('HP-GL interpreter V3.5 by Bernd Bock');
  CreatePbm := false;
  
  (* get the name of the HP-GL file *)
  if ParamCount > 0 then begin
    FileName := ParamStr(1); (* first commandline parameter treated as filename *)
    GotoXY(1, 5);
  end
  else begin
    DrawBox( 1, 4, 52, 3);
    DrawBox(53, 4, 38, 3);
    DrawBox( 1, 7, 52, 3);
    DrawBox(53, 7, 38, 3);
    GotoXY(3, 8);
    Write('Enter PBM output filename:');
    GotoXY(55, 8);
    Write('AR correction? (Y/N)');
    GotoXY(3, 5);
    Write('Enter name of HPGL file to draw: ');
    FileName := InputFileName;
    if Length(FileName) = 0 then goto byebye; (* abort if no filename entered *)
    GotoXY(36, 8);
    FileNamePBM := InputFileName;
    if Length(FileNamePBM) > 0 then begin
      CreatePbm := true;
      GotoXY(76, 8);
      repeat
        ARCorr := UPCASE(InputChar);
      until (ARCorr = 'Y') or (ARCorr = 'N');
      GotoXY(76, 8);
      Write(ARCorr);
    end;
    GotoXY(1, 10);
  end;
  (* open the HP-GL file *)
  Assign(FilVar, FileName);
  (*$I-*)
  Reset(FilVar);
  (*$I+*)
  if IOResult <> 0 then
  begin
    WriteLn('File not found: ', FileName);
    goto byebye;
  end;

  (* initialization *)
  Break := false;
  State := START;
  ParamName := 'X';
  LastX := 0;
  LastY := 0;
  MinX := MAX_COORD;
  MaxX := MIN_COORD;
  MinY := MAX_COORD;
  MaxY := MIN_COORD;
  DotCnt := 10;

  HideCursor;
  WriteLn;

  Pass1;
  Pass2;

  Close(FilVar);

  (* save picture as PBM file *)
  if CreatePbm then SaveScreenToPBM(FileNamePBM, (ARCorr = 'Y'));
  
  Beep;
  WaitForKey;
  
byebye:
  ShowCursor;
end.
